import 'dart:io';

import 'package:device_info_plus/device_info_plus.dart';
import 'package:eschool/cubits/appLocalizationCubit.dart';
import 'package:eschool/cubits/downloadFileCubit.dart';
import 'package:eschool/cubits/schoolConfigurationCubit.dart';
import 'package:eschool/data/models/studyMaterial.dart';
import 'package:eschool/data/repositories/subjectRepository.dart';
import 'package:eschool/ui/widgets/downloadFileBottomsheetContainer.dart';
import 'package:eschool/ui/widgets/errorMessageOverlayContainer.dart';
import 'package:eschool/utils/constants.dart';
import 'package:eschool/utils/errorMessageKeysAndCodes.dart';
import 'package:eschool/utils/labelKeys.dart';
import 'package:eschool/utils/systemModules.dart';
import 'package:flutter/material.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:get/get.dart';
import 'package:intl/intl.dart' as intl;
import 'package:package_info_plus/package_info_plus.dart';
import 'package:path_provider/path_provider.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:url_launcher/url_launcher.dart';
import 'package:open_file/open_file.dart';

// ignore: avoid_classes_with_only_static_members
class Utils {
  //This extra padding will add to MediaQuery.of(context).padding.top in orderto give same top padding in every screen

  static double screenContentTopPadding = 15.0;
  static double screenContentHorizontalPadding = 25.0;
  static double screenTitleFontSize = 18.0;
  static double screenOnbordingTitleFontSize = 25.0;
  static double screenContentHorizontalPaddingInPercentage = 0.075;

  static double screenSubTitleFontSize = 14.0;
  static double extraScreenContentTopPaddingForScrolling = 0.0275;
  static double appBarSmallerHeightPercentage = 0.15;

  static double appBarMediumtHeightPercentage = 0.18;

  static double bottomNavigationHeightPercentage = 0.08;
  static double bottomNavigationBottomMargin = 25;

  static double appBarBiggerHeightPercentage = 0.215;
  static double appBarContentTopPadding = 25.0;
  static double bottomSheetTopRadius = 20.0;
  static double subjectFirstLetterFontSize = 20;

  static double defaultProfilePictureHeightAndWidthPercentage = 0.175;

  static double questionContainerHeightPercentage = 0.725;

  static Duration tabBackgroundContainerAnimationDuration =
      const Duration(milliseconds: 300);

  static Duration showCaseDisplayDelayInDuration =
      const Duration(milliseconds: 350);
  static Curve tabBackgroundContainerAnimationCurve = Curves.easeInOut;

  static double shimmerLoadingContainerDefaultHeight = 7;

  static int defaultShimmerLoadingContentCount = 6;

  // static GlobalKey<NavigatorState> rootNavigatorKey =
  //     GlobalKey<NavigatorState>();

  static final List<String> weekDays = [
    mondayKey,
    tuesdayKey,
    wednesdayKey,
    thursdayKey,
    fridayKey,
    saturdayKey,
    sundayKey
  ];

  static final List<String> weekDaysFullForm = [
    "Monday",
    "Tuesday",
    "Wednesday",
    "Thursday",
    "Friday",
    "Saturday",
    "Sunday"
  ];

  //to give bottom scroll padding in screen where
  //bottom navigation bar is displayed
  static double getScrollViewBottomPadding(BuildContext context) {
    return MediaQuery.of(context).size.height *
            (Utils.bottomNavigationHeightPercentage) +
        Utils.bottomNavigationBottomMargin * (1.5);
  }

  //to give top scroll padding to screen content
  //
  static double getScrollViewTopPadding({
    required BuildContext context,
    required double appBarHeightPercentage,
  }) {
    return MediaQuery.of(context).size.height *
        (appBarHeightPercentage + extraScreenContentTopPaddingForScrolling);
  }

  static String getImagePath(String imageName) {
    return "assets/images/$imageName";
  }

  static ColorScheme getColorScheme(BuildContext context) {
    return Theme.of(context).colorScheme;
  }

  static Locale getLocaleFromLanguageCode(String languageCode) {
    List<String> result = languageCode.split("-");
    return result.length == 1
        ? Locale(result.first)
        : Locale(result.first, result.last);
  }

  static String getTranslatedLabel(String labelKey) {
    return labelKey.tr.trim();
  }

  static Future<dynamic> showBottomSheet({
    required Widget child,
    required BuildContext context,
    bool? enableDrag,
  }) async {
    final result = await Get.bottomSheet(
      // Prevent
      isDismissible: true,
      child,
      enableDrag: enableDrag ?? false,
      isScrollControlled: true,
      clipBehavior: Clip.antiAliasWithSaveLayer,
      shape: RoundedRectangleBorder(
        borderRadius: BorderRadius.only(
          topLeft: Radius.circular(bottomSheetTopRadius),
          topRight: Radius.circular(bottomSheetTopRadius),
        ),
      ),
    );
    return result;
  }

  static bool isTodayInSessionYear(DateTime firstDate, DateTime lastDate) {
    final currentDate = DateTime.now();

    return (currentDate.isAfter(firstDate) && currentDate.isBefore(lastDate)) ||
        isSameDay(firstDate) ||
        isSameDay(lastDate);
  }

  static bool isSameDay(DateTime dateTime) {
    final currentDate = DateTime.now();
    return (currentDate.day == dateTime.day) &&
        (currentDate.month == dateTime.month) &&
        (currentDate.year == dateTime.year);
  }

  static String getMonthName(int monthNumber) {
    return months[monthNumber - 1];
  }

  static int getMonthNumber(String monthName) {
    return (months.indexWhere((element) => element == monthName)) + 1;
  }

  static List<String> buildMonthYearsBetweenTwoDates(
    DateTime startDate,
    DateTime endDate,
  ) {
    List<String> dateTimes = [];
    DateTime current = startDate;
    while (current.difference(endDate).isNegative) {
      current = current.add(const Duration(days: 24));
      dateTimes.add("${getMonthName(current.month)}, ${current.year}");
    }
    return dateTimes.toSet().toList();
  }

  static String formatTime(String time) {
    final hourMinuteSecond = time.split(":");
    final hour = int.parse(hourMinuteSecond.first) < 13
        ? int.parse(hourMinuteSecond.first)
        : int.parse(hourMinuteSecond.first) - 12;
    final amOrPm = int.parse(hourMinuteSecond.first) > 12 ? "PM" : "AM";
    return "${hour.toString().padLeft(2, '0')}:${hourMinuteSecond[1]} $amOrPm";
  }

  static String extractTimeFromDateString(String dateString) {
    try {
      final parts = dateString.split(' ');
      if (parts.length >= 2) {
        String timePart = parts[1];
        if (parts.length >= 3) {
          timePart += " ${parts[2]}";
        }
        return timePart;
      }
    } catch (e) {
      // If parsing fails, return the original string
    }
    return dateString;
  }

  static String formatApiDateTime(String dateTimeString) {
    try {
      if (dateTimeString.contains('/') && dateTimeString.contains(' ')) {
        final parts = dateTimeString.split(' ');
        if (parts.length >= 2) {
          final timeParts = parts.sublist(1);
          return timeParts.join(' ');
        }
      }
      return dateTimeString;
    } catch (e) {
      print('Error formatting datetime: $dateTimeString, Error: $e');
      return dateTimeString;
    }
  }

  static String formatAssignmentDueDate(
    DateTime dateTime,
    BuildContext context,
  ) {
    final monthName = Utils.getMonthName(dateTime.month);
    final hour = dateTime.hour < 13 ? dateTime.hour : dateTime.hour - 12;
    final amOrPm = hour > 12 ? "PM" : "AM";
    return "${Utils.getTranslatedLabel(dueKey)}, ${dateTime.day} $monthName ${dateTime.year}, ${hour.toString().padLeft(2, '0')}:${dateTime.minute.toString().padLeft(2, '0')} $amOrPm";
  }

  static Future<void> showCustomSnackBar({
    required BuildContext context,
    required String errorMessage,
    required Color backgroundColor,
    Duration delayDuration = errorMessageDisplayDuration,
  }) async {
    OverlayState? overlayState = Overlay.of(context);
    OverlayEntry overlayEntry = OverlayEntry(
      builder: (context) => ErrorMessageOverlayContainer(
        backgroundColor: backgroundColor,
        errorMessage: errorMessage,
      ),
    );

    overlayState.insert(overlayEntry);
    await Future.delayed(delayDuration);
    overlayEntry.remove();
  }

  static String getErrorMessageFromErrorCode(
    BuildContext context,
    String errorCode,
  ) {
    return Utils.getTranslatedLabel(
      ErrorMessageKeysAndCode.getErrorMessageKeyFromCode(errorCode),
    );
  }

  static double appContentTopScrollPadding({required BuildContext context}) {
    return kToolbarHeight + MediaQuery.of(context).padding.top;
  }

  //0 = Pending/In Review , 1 = Accepted , 2 = Rejected
  static String getAssignmentSubmissionStatusKey(int status) {
    if (status == 0) {
      return inReviewKey;
    }
    if (status == 1) {
      return acceptedKey;
    }
    if (status == 2) {
      return rejectedKey;
    }
    if (status == 3) {
      return resubmittedKey;
    }
    return "";
  }

  static String getBackButtonPath(BuildContext context) {
    return Directionality.of(context).name == TextDirection.rtl.name
        ? getImagePath("rtl_back_icon.svg")
        : getImagePath("back_icon.svg");
  }

  static Future<String?> _checkIfFileExists({
    required StudyMaterial studyMaterial,
    required bool storeInExternalStorage,
  }) async {
    try {
      // Always check external storage first (priority for permanent files)
      if (await hasStoragePermissionGiven()) {
        final externalDirectory = Platform.isAndroid
            ? await getExternalStorageDirectory()
            : await getApplicationDocumentsDirectory();
        final externalFilePath =
            "${externalDirectory?.path}/${studyMaterial.fileName}.${studyMaterial.fileExtension}";

        final externalFile = File(externalFilePath);
        if (await externalFile.exists()) {
          return externalFilePath; // Found in external storage
        }
      }

      // If not found in external storage, check temporary directory
      final tempDir = await getTemporaryDirectory();
      final tempFilePath =
          "${tempDir.path}/${studyMaterial.fileName}.${studyMaterial.fileExtension}";

      final tempFile = File(tempFilePath);
      if (await tempFile.exists()) {
        return tempFilePath; // Found in temp storage
      }

      return null; // File not found in either location
    } catch (e) {
      return null;
    }
  }

  static Future<void> openDownloadBottomsheet({
    required BuildContext context,
    required bool storeInExternalStorage,
    required StudyMaterial studyMaterial,
  }) async {
    // First check if file already exists
    final existingFilePath = await _checkIfFileExists(
      studyMaterial: studyMaterial,
      storeInExternalStorage: storeInExternalStorage,
    );

    if (existingFilePath != null) {
      // File already exists, open it directly
      try {
        OpenFile.open(existingFilePath);
      } catch (e) {
        // If opening fails, show error message
        showCustomSnackBar(
          context: context,
          errorMessage: "Could not open file: ${e.toString()}",
          backgroundColor: Theme.of(context).colorScheme.error,
        );
      }
      return;
    }

    // File doesn't exist, show download bottom sheet
    showBottomSheet(
      child: BlocProvider<DownloadFileCubit>(
        create: (context) => DownloadFileCubit(SubjectRepository()),
        child: DownloadFileBottomsheetContainer(
          storeInExternalStorage: storeInExternalStorage,
          studyMaterial: studyMaterial,
        ),
      ),
      context: context,
    ).then((result) {
      if (result != null) {
        if (result['error']) {
          print("This is the result $result");
          print("This is the result ${result['error']}");
          showCustomSnackBar(
            context: context,
            errorMessage: getErrorMessageFromErrorCode(
              context,
              result['message'].toString(),
            ),
            backgroundColor: Theme.of(context).colorScheme.error,
          );
        } else {
          try {
            OpenFile.open(result['filePath'].toString());
          } catch (e, st) {
            print("this is the error $e");
            print("this is the st $st");
            showCustomSnackBar(
              context: context,
              errorMessage: getTranslatedLabel(
                storeInExternalStorage
                    ? fileDownloadedSuccessfullyKey
                    : unableToOpenKey,
              ),
              backgroundColor: Theme.of(context).colorScheme.error,
            );
          }
        }
      }
    });
  }

  static intl.DateFormat hourMinutesDateFormat = intl.DateFormat.jm();

  static String formatDateAndTime(DateTime dateTime) {
    return intl.DateFormat("dd-MM-yyyy  kk:mm").format(dateTime);
  }

  static String formatDate(DateTime dateTime) {
    return "${dateTime.day.toString().padLeft(2, '0')}-${dateTime.month.toString().padLeft(2, '0')}-${dateTime.year}";
  }

  static DateTime? parseApiDate(String dateString) {
    try {
      // Handle date formats with time (any admin format + HH:mm AM/PM)
      if (dateString.contains(' ') && dateString.contains('M')) {
        List<String> dateTimeParts = dateString.split(' ');
        if (dateTimeParts.length >= 3) {
          String datePart = dateTimeParts[0];
          String timePart = dateTimeParts[1];
          String amPmPart = dateTimeParts[2];

          // Try to detect and parse the date format automatically
          DateTime? parsedDate = _detectAndParseDate(datePart);
          if (parsedDate != null) {
            List<String> timeComponents = timePart.split(':');
            if (timeComponents.length == 2) {
              int hour = int.parse(timeComponents[0]);
              int minute = int.parse(timeComponents[1]);

              if (amPmPart.toUpperCase() == 'PM' && hour != 12) {
                hour += 12;
              } else if (amPmPart.toUpperCase() == 'AM' && hour == 12) {
                hour = 0;
              }

              return DateTime(parsedDate.year, parsedDate.month, parsedDate.day,
                  hour, minute);
            }
          }
        }
      }

      // Handle MM/dd/yyyy HH:mm AM/PM format
      if (dateString.contains('/') && dateString.contains(' ')) {
        List<String> dateTimeParts = dateString.split(' ');
        if (dateTimeParts.length >= 3) {
          String datePart = dateTimeParts[0];
          String timePart = dateTimeParts[1];
          String amPmPart = dateTimeParts[2];

          List<String> dateComponents = datePart.split('/');
          if (dateComponents.length == 3) {
            int month = int.parse(dateComponents[0]);
            int day = int.parse(dateComponents[1]);
            int year = int.parse(dateComponents[2]);

            List<String> timeComponents = timePart.split(':');
            if (timeComponents.length == 2) {
              int hour = int.parse(timeComponents[0]);
              int minute = int.parse(timeComponents[1]);

              if (amPmPart.toUpperCase() == 'PM' && hour != 12) {
                hour += 12;
              } else if (amPmPart.toUpperCase() == 'AM' && hour == 12) {
                hour = 0;
              }

              return DateTime(year, month, day, hour, minute);
            }
          }
        }
      }

      // Handle MM/dd/yyyy format without time
      if (dateString.contains('/') && !dateString.contains(' ')) {
        List<String> parts = dateString.split('/');
        if (parts.length == 3) {
          int month = int.parse(parts[0]);
          int day = int.parse(parts[1]);
          int year = int.parse(parts[2]);
          return DateTime(year, month, day);
        }
      }

      // Handle date-only formats (any admin format without time)
      if (!dateString.contains(' ')) {
        return _detectAndParseDate(dateString);
      }

      return DateTime.parse(dateString);
    } catch (e) {
      print('Error parsing date: $dateString, Error: $e');
      return null;
    }
  }

  static String formatApiDate(String dateString) {
    return dateString.isEmpty ? '--' : dateString;
  }

  static String dateConverter(
    DateTime myEndDate,
    BuildContext contxt,
    bool fromResult,
  ) {
    String date;

    final formattedDate = intl.DateFormat('dd MMM, yyyy',
            contxt.read<AppLocalizationCubit>().state.language.languageCode)
        .add_jm()
        .format(myEndDate);

    final formattedTime = intl.DateFormat('hh:mm a').format(myEndDate);
    //check for today or tomorrow or specific date
    final now = DateTime.now();
    final today = DateTime(now.year, now.month, now.day);
    final tomorrow = today.add(Duration(days: 1));
    final checkEndDate =
        DateTime(myEndDate.year, myEndDate.month, myEndDate.day);

    if (checkEndDate == today) {
      date = fromResult
          ? "${Utils.getTranslatedLabel(submittedKey)} : ${Utils.getTranslatedLabel(todayKey)}" //
          : "${Utils.getTranslatedLabel(todayKey)}, $formattedTime";
    } else if (checkEndDate == tomorrow) {
      date = fromResult
          ? "${Utils.getTranslatedLabel(submittedKey)} : ${Utils.getTranslatedLabel(tomorrowKey)}"
          : "${Utils.getTranslatedLabel(tomorrowKey)}, $formattedTime";
    } else {
      date = fromResult
          ? '${Utils.getTranslatedLabel(submittedKey)} : ${formattedDate}'
          : '$formattedDate';
    }
    return date;
  }

  //It will return - if given value is empty
  static String formatEmptyValue(String value) {
    return value.isEmpty ? "-" : value;
  }

  static Future<bool> forceUpdate(String updatedVersion) async {
    PackageInfo packageInfo = await PackageInfo.fromPlatform();
    String currentVersion = "${packageInfo.version}+${packageInfo.buildNumber}";
    if (updatedVersion.isEmpty) {
      return false;
    }

    final bool updateBasedOnVersion = _shouldUpdateBasedOnVersion(
      currentVersion.split("+").first,
      updatedVersion.split("+").first,
    );

    if (updatedVersion.split("+").length == 1 ||
        currentVersion.split("+").length == 1) {
      return updateBasedOnVersion;
    }

    final bool updateBasedOnBuildNumber = _shouldUpdateBasedOnBuildNumber(
      currentVersion.split("+").last,
      updatedVersion.split("+").last,
    );

    return updateBasedOnVersion || updateBasedOnBuildNumber;
  }

  static bool _shouldUpdateBasedOnVersion(
    String currentVersion,
    String updatedVersion,
  ) {
    List<int> currentVersionList =
        currentVersion.split(".").map((e) => int.parse(e)).toList();
    List<int> updatedVersionList =
        updatedVersion.split(".").map((e) => int.parse(e)).toList();

    if (updatedVersionList[0] > currentVersionList[0]) {
      return true;
    }
    if (updatedVersionList[1] > currentVersionList[1]) {
      return true;
    }
    if (updatedVersionList[2] > currentVersionList[2]) {
      return true;
    }

    return false;
  }

  static bool _shouldUpdateBasedOnBuildNumber(
    String currentBuildNumber,
    String updatedBuildNumber,
  ) {
    return int.parse(updatedBuildNumber) > int.parse(currentBuildNumber);
  }

  static String getLottieAnimationPath(String animationName) {
    return "assets/animations/$animationName";
  }

  static void showFeatureDisableInDemoVersion(BuildContext context) {
    showCustomSnackBar(
      context: context,
      errorMessage: Utils.getTranslatedLabel(featureDisableInDemoVersionKey),
      backgroundColor: Theme.of(context).colorScheme.error,
    );
  }

  //0 = Pending , 1 = Paid, 2 = Partially Paid ˆset according to API response.
  static String getStudentFeesStatusKey(int status) {
    if (status == 0) {
      return pendingKey;
    }
    if (status == 1) {
      return paidKey;
    }
    if (status == 2) {
      return partiallyPaidKey;
    }
    return "";
  }

  static bool isModuleEnabled(
      {required BuildContext context, required String moduleId}) {
    final enabledFeatures = context
        .read<SchoolConfigurationCubit>()
        .getSchoolConfiguration()
        .enabledModules;

    //Module id will have "1" or "1#2".
    final ids = moduleId.split("$moduleIdJoiner").toList();
    if (ids.contains(defaultModuleId.toString())) {
      return true;
    }

    bool featureEnabled = false;
    for (var i = 0; i < ids.length; i++) {
      if (enabledFeatures.containsKey(ids[i].toString())) {
        featureEnabled = true;
        break;
      }
    }

    //
    return featureEnabled;
  }

  static Future<bool> hasStoragePermissionGiven() async {
    if (Platform.isIOS) {
      bool permissionGiven = await Permission.storage.isGranted;
      if (!permissionGiven) {
        permissionGiven = (await Permission.storage.request()).isGranted;
        return permissionGiven;
      }
      return permissionGiven;
    }

    //if it is for android
    final deviceInfoPlugin = DeviceInfoPlugin();
    final androidDeviceInfo = await deviceInfoPlugin.androidInfo;
    if (androidDeviceInfo.version.sdkInt < 33) {
      bool permissionGiven = await Permission.storage.isGranted;
      if (!permissionGiven) {
        permissionGiven = (await Permission.storage.request()).isGranted;
        return permissionGiven;
      }
      return permissionGiven;
    } else {
      bool permissionGiven = await Permission.photos.isGranted;
      if (!permissionGiven) {
        permissionGiven = (await Permission.photos.request()).isGranted;
        return permissionGiven;
      }
      return permissionGiven;
    }
  }

  static Future<bool> hasCameraPermissionGiven() async {
    bool permissionGiven = await Permission.camera.isGranted;
    if (!permissionGiven) {
      permissionGiven = (await Permission.camera.request()).isGranted;
      return permissionGiven;
    }
    return permissionGiven;
  }

  static Future<bool> hasGalleryPermissionGiven() async {
    bool permissionGiven = await Permission.photos.isGranted;
    if (!permissionGiven) {
      permissionGiven = (await Permission.photos.request()).isGranted;
      return permissionGiven;
    }
    return permissionGiven;
  }

  static DateTime? parseAssignmentDueDate(
      String dateString, String dateFormat) {
    try {
      if (dateString.contains(' ')) {
        List<String> parts = dateString.split(' ');
        if (parts.length >= 3) {
          String datePart = parts[0];
          String timePart = parts[1];
          String amPmPart = parts[2];

          DateTime? dateOnly = _parseDateByFormat(datePart, dateFormat);
          if (dateOnly == null) return null;

          List<String> timeComponents = timePart.split(':');
          if (timeComponents.length == 2) {
            int hour = int.parse(timeComponents[0]);
            int minute = int.parse(timeComponents[1]);

            if (amPmPart.toUpperCase() == 'PM' && hour != 12) {
              hour += 12;
            } else if (amPmPart.toUpperCase() == 'AM' && hour == 12) {
              hour = 0;
            }

            return DateTime(
                dateOnly.year, dateOnly.month, dateOnly.day, hour, minute);
          }
        }
      }
      return _parseDateByFormat(dateString, dateFormat);
    } catch (e) {
      print(
          'Error parsing assignment due date: $dateString with format: $dateFormat, Error: $e');
      return null;
    }
  }

  static DateTime? _parseDateByFormat(String dateString, String format) {
    try {
      switch (format) {
        case 'Y-m-d':
          List<String> parts = dateString.split('-');
          if (parts.length == 3) {
            int year = int.parse(parts[0]);
            int month = int.parse(parts[1]);
            int day = int.parse(parts[2]);
            return DateTime(year, month, day);
          }
          break;
        case 'd/m/Y':
          List<String> parts = dateString.split('/');
          if (parts.length == 3) {
            int day = int.parse(parts[0]);
            int month = int.parse(parts[1]);
            int year = int.parse(parts[2]);
            return DateTime(year, month, day);
          }
          break;
        case 'm/d/Y':
          List<String> parts = dateString.split('/');
          if (parts.length == 3) {
            int month = int.parse(parts[0]);
            int day = int.parse(parts[1]);
            int year = int.parse(parts[2]);
            return DateTime(year, month, day);
          }
          break;
        case 'Y/m/d':
          List<String> parts = dateString.split('/');
          if (parts.length == 3) {
            int year = int.parse(parts[0]);
            int month = int.parse(parts[1]);
            int day = int.parse(parts[2]);
            return DateTime(year, month, day);
          }
          break;
        case 'Y/d/m':
          List<String> parts = dateString.split('/');
          if (parts.length == 3) {
            int year = int.parse(parts[0]);
            int day = int.parse(parts[1]);
            int month = int.parse(parts[2]);
            return DateTime(year, month, day);
          }
          break;
        case 'm-d-Y':
          List<String> parts = dateString.split('-');
          if (parts.length == 3) {
            int month = int.parse(parts[0]);
            int day = int.parse(parts[1]);
            int year = int.parse(parts[2]);
            return DateTime(year, month, day);
          }
          break;
        case 'd-m-Y':
          List<String> parts = dateString.split('-');
          if (parts.length == 3) {
            int day = int.parse(parts[0]);
            int month = int.parse(parts[1]);
            int year = int.parse(parts[2]);
            return DateTime(year, month, day);
          }
          break;
        case 'Y-d-m':
          List<String> parts = dateString.split('-');
          if (parts.length == 3) {
            int year = int.parse(parts[0]);
            int day = int.parse(parts[1]);
            int month = int.parse(parts[2]);
            return DateTime(year, month, day);
          }
          break;
        default:
          return DateTime.parse(dateString);
      }
    } catch (e) {
      print('Error parsing date with format $format: $dateString, Error: $e');
    }
    return null;
  }

  /// Enhanced date parsing that handles configurable formats from admin panel
  static DateTime? parseApiDateWithFormat(String dateString,
      {String? dateFormat}) {
    if (dateString.isEmpty) return null;

    try {
      // If we have a specific date format from the API, use it
      if (dateFormat != null && dateFormat.isNotEmpty) {
        return _parseWithSpecificFormat(dateString, dateFormat);
      }

      // Fallback to general parsing for backward compatibility
      return parseApiDate(dateString);
    } catch (e) {
      print(
          'Error parsing date: $dateString with format: $dateFormat, Error: $e');
      return null;
    }
  }

  /// Automatically detect and parse date format from common admin formats
  static DateTime? _detectAndParseDate(String dateString) {
    if (dateString.isEmpty) return null;

    // List of all supported admin date formats
    final supportedFormats = [
      'Y-m-d', // 2025-07-28
      'm-d-Y', // 07-28-2025
      'd-m-Y', // 28-07-2025
      'Y-d-m', // 2025-28-07
      'Y/m/d', // 2025/07/28
      'm/d/Y', // 07/28/2025
      'd/m/Y', // 28/07/2025
      'Y/d/m', // 2025/28/07
    ];

    // Try each format until one works
    for (String format in supportedFormats) {
      try {
        DateTime? result = _parseDateByFormat(dateString, format);
        if (result != null) {
          return result;
        }
      } catch (e) {
        // Continue to next format
        continue;
      }
    }

    return null;
  }

  /// Parse date with specific format provided by admin panel
  static DateTime? _parseWithSpecificFormat(String dateString, String format) {
    try {
      if (!dateString.contains(' ')) {
        // Date only format
        return _parseDateByFormat(dateString, format);
      }

      // Date with time format
      List<String> parts = dateString.split(' ');
      if (parts.length >= 3) {
        String datePart = parts[0];
        String timePart = parts[1];
        String amPmPart = parts[2];

        DateTime? dateOnly = _parseDateByFormat(datePart, format);
        if (dateOnly == null) return null;

        // Parse time part
        List<String> timeComponents = timePart.split(':');
        if (timeComponents.length == 2) {
          int hour = int.parse(timeComponents[0]);
          int minute = int.parse(timeComponents[1]);

          // Handle AM/PM
          if (amPmPart.toUpperCase() == 'PM' && hour != 12) {
            hour += 12;
          } else if (amPmPart.toUpperCase() == 'AM' && hour == 12) {
            hour = 0;
          }

          return DateTime(
              dateOnly.year, dateOnly.month, dateOnly.day, hour, minute);
        }
      }

      return null;
    } catch (e) {
      print(
          'Error parsing formatted date: $dateString with format: $format, Error: $e');
      return null;
    }
  }

  /// Launch phone dialer with the given phone number
  static Future<void> launchPhoneDialer(String phoneNumber) async {
    if (phoneNumber.isEmpty) return;

    // Clean the phone number (remove spaces, dashes, etc.)
    final cleanedNumber = phoneNumber.replaceAll(RegExp(r'[^0-9+]'), '');
    final uri = Uri(scheme: 'tel', path: cleanedNumber);

    try {
      if (await canLaunchUrl(uri)) {
        await launchUrl(uri);
      } else {
        throw 'Could not launch phone dialer';
      }
    } catch (e) {
      print('Error launching phone dialer: $e');
    }
  }
}

extension DateTimeExtension on DateTime {
  bool isSameDayAs(DateTime other) =>
      this.day == other.day &&
      this.month == other.month &&
      this.year == other.year;

  String get relativeFormatedDate {
    final today = DateTime.now();
    final yesterday = today.subtract(const Duration(days: 1));

    if (this.isSameDayAs(today)) {
      return "today";
    } else if (this.isSameDayAs(yesterday)) {
      return "yesterday";
    } else {
      return intl.DateFormat('d MMMM yyyy').format(this);
    }
  }
}

extension EmptyPadding on num {
  SizedBox get sizedBoxHeight => SizedBox(height: toDouble());
  SizedBox get sizedBoxWidth => SizedBox(width: toDouble());
}
